#include <stdio.h>
#include <stdlib.h>
#include <endian.h>
#include <byteswap.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <string.h>

#define BLOCKSIZE 2352
#define DEF_THRESHOLD 0

int opt_writefile=1;
int opt_verbose=1;

int threshold = DEF_THRESHOLD;

void
usage(const char* name)
{
  printf("usage: %s filename\n", name);
}

int
issilent(signed short* buf, int block, int thresh)
{
  int i;
  buf += block*(BLOCKSIZE/sizeof(short));
  for(i=0; i<(BLOCKSIZE/sizeof(short)); i++) {
#if BYTE_ORDER == LITTLE_ENDIAN
    register signed short b=bswap_16(buf[i]);
#else
    register signed short b=buf[i];
#endif
    if (abs(b)>thresh) return 0;
  }
  return 1;
}

char*
outfname(const char* infname)
{
  char* buf=(char*)malloc(strlen(infname)+2);
  char* slashpos = strrchr(infname, '/');
  if(slashpos) {
    strncpy(buf, infname, (slashpos-infname)+1);
    strcat(buf, "T_");
    strcat(buf, slashpos+1);
  }
  return buf;
}

int
main(int argc, char *argv[])
{
  int infd;
  signed short* inbuf;
  struct stat stbuf;
  int startskip, endskip, nblocks;
  char *inname, *outname=(char*)0;
  char opt;

  if(argc<2) {
    usage(argv[0]);
    return 1;
  }

  while((opt=getopt(argc,argv,"no:qt:v"))!=EOF) {
   switch(opt) {
   case 'n': opt_writefile = 0;  break;
   case 'o': outname = optarg; break;
   case 'q': opt_verbose = 0; break;
   case 't': threshold = atoi(optarg); break;
   case 'v': opt_verbose++; break;
   }
  }
  
  inname=argv[optind];

  infd = open(inname, O_RDONLY);
  if(infd <= 0) {
    fprintf(stderr, "%s: cannot open file %s\n", argv[0],inname);
    return 1;
  }  
  fstat(infd, &stbuf);
  (void *)inbuf = mmap(0, stbuf.st_size, PROT_READ, MAP_PRIVATE, infd, 0);
  nblocks = stbuf.st_size/BLOCKSIZE;
  for(startskip=0; issilent(inbuf, startskip, threshold); startskip++) ;
  for(endskip=0; issilent(inbuf, nblocks-(endskip+1), threshold); endskip++) ;

  if(opt_verbose) {
    printf("%s: %d leading/%d trailing blocks of silence; ", inname, startskip, endskip);
    printf("total: %.04f sec.\n", (startskip+endskip)/75.0);
  }
  if(opt_writefile) {
    int ofd;
    char* of = outname?strdup(outname):outfname(inname);
    int written, outsize;
    ofd = creat(of, stbuf.st_mode);
    if(ofd<=0) {
      fprintf(stderr, "%s: cannot create file %s\n", argv[0], of);
      exit(3);
    }
    outsize = stbuf.st_size-((startskip+endskip)*BLOCKSIZE);
    written = write(ofd, ((char*)inbuf)+(startskip*BLOCKSIZE), outsize);
    if(written<outsize) {
      fprintf(stderr, "%s: could only write %d of %d bytes\n", of, written, outsize);
    }
    close(ofd);
    free(of);
  }
  munmap((void*)inbuf, stbuf.st_size);
  close(infd);
  return 0;
}
